<?php
// FILE: admin/generate_temporary_barcodes.php
require_once __DIR__ . '/../init.php';

// 🛑 التحقق من صلاحيات المدير
if (!isset($_SESSION['user']) || ($_SESSION['user']['role'] !== 'super_admin' && $_SESSION['user']['role'] !== 'admin')) {
    // تأكد من أن $config مُعرفة في init.php أو استخدم مسار ثابت
    $base_url = defined('BASE_URL') ? BASE_URL : '/barcode-system/';
    header('Location: ' . $base_url . 'admin/index.php');
    exit;
}

// ⚙️ إعدادات توليد الباركود
$total_barcodes = 100000; // يمكن تغيير هذا العدد

// 🆕 قائمة البادئات المتاحة (تم تغيير ACC إلى AC للمواءمة مع المثال الأصلي ولكن سنستخدم ACC كما ورد في الكود المرفق)
$available_prefixes = ['BC', 'RE', 'BE', 'ACC'];

// 🆕 استقبال البادئة المختارة
$selected_prefix = 'BC'; // القيمة الافتراضية
if (isset($_GET['prefix']) && in_array($_GET['prefix'], $available_prefixes)) {
    $selected_prefix = $_GET['prefix'];
}

// بناء البادئة النهائية
$prefix = $selected_prefix . date('ymd') ;  // البادئة المختارة + تاريخ اليوم YYMMDD


// 🆕 متغيرات التحكم في الطباعة
// رقم البداية: القيمة الافتراضية 1 أو القيمة المدخلة من المستخدم
$start_number_input = isset($_GET['start']) ? (int)$_GET['start'] : 1;
if ($start_number_input < 1) $start_number_input = 1;

// عدد الصفحات المراد دمجها وعرضها للطباعة
$print_pages_count = isset($_GET['print_count']) ? (int)$_GET['print_count'] : 1;
if ($print_pages_count < 1) $print_pages_count = 1;
// الحد الأقصى للطباعة المدمجة
if ($print_pages_count > 100) $print_pages_count = 100; 

// 🛑 حساب رقم البداية الفعلي
$limit = 100; // 100 باركود في الصفحة الواحدة
$total_pages = ceil($total_barcodes / $limit);

$page = isset($_GET['p']) ? (int)$_GET['p'] : 1;
if ($page < 1) $page = 1;
if ($page > $total_pages) $page = $total_pages;

// حساب إجمالي عدد الباركودات المراد عرضها الآن
$total_limit = $limit * $print_pages_count;

// حساب رقم البداية الفعلي لـ *أول* صفحة
$current_page_index = (($page - 1) * $limit);

// رقم البداية الفعلي لتوليد الباركود
$actual_start_number = $start_number_input + $current_page_index;

$barcodes = [];
// توليد الباركودات المطلوبة لجميع الصفحات المحددة
for ($i = 0; $i < $total_limit; $i++) {
    $current_number = $actual_start_number + $i;
    
    // توقف إذا تجاوزنا العدد الإجمالي المطلوب (من نقطة البداية)
    if (($current_number - $start_number_input) >= $total_barcodes) break;
    
    // تنسيق الرقم ليكون طوله ثابتاً (7 أرقام: BC0000001)
    $number_string = str_pad($current_number, 7, '0', STR_PAD_LEFT);
    $barcodes[] = $prefix . $number_string;
}

?>
<!doctype html>
<html lang="ar">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width,initial-scale=1">
    <title>طباعة باركودات غير مسجلة</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/jsbarcode@3.11.6/dist/JsBarcode.all.min.js"></script>
<style>
        /* 1. 🎨 تحسين الخطوط والألوان العامة */
        body { 
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif; 
            direction: rtl; 
            padding: 20px; 
            background-color: #f8f9fa;
        }
        
        /* 2. 🏷️ تصميم الملصق (للعرض على الشاشة) */
        .barcode-container {
            display: inline-flex;
            margin: 5px;
            text-align: center;
            border: none;
            background-color: #ffffff;
            padding: 0;
            width: 140mm; 
            height: 70mm; 
            max-width: 100%;
            border-radius: 8px; 
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
            flex-direction: row; 
            justify-content: space-between;
            overflow: hidden;
            position: relative;
        }

        /* 3. 🖼️ قسم الباركود */
        .barcode-section {
            width: 70mm; 
            height: 100%; 
            display: flex;
            flex-direction: column;
            justify-content: center;
            align-items: center;
            border-left: 2px dashed #e0e0e0; /* الشريط الفاصل للعرض */
            padding: 2mm;
            background-color: #fefefe;
        }

        .barcode-section svg {
            width: 90%; 
            height: auto; 
            display: block;
            margin: 0;
            padding: 0;
        }
        
        .barcode-text {
            font-size: 16px;
            margin-top: 3px;
            font-weight: 700;
            color: #333;
            letter-spacing: 1px;
        }
        
        /* 4. 📝 قسم التعليمات */
        .instructions-section {
            width: 70mm; 
            height: 100%; 
            display: flex;
            flex-direction: column;
            justify-content: center;
            text-align: right;
            padding: 2mm 6mm 2mm 0.2mm;
            background-color: #f7f7f7; /* خلفية قسم التعليمات للعرض */
            font-size: 10px;
            line-height: 1.3;
            color: #555;
        }

        .instructions-section h6 {
            font-size: 12px;
            font-weight: 700;
            color: #007bff; /* لون عنوان التعليمات للعرض */
            margin-bottom: 5px;
            border-bottom: 1px solid #ddd;
            padding-bottom: 2px;
        }
        
        .instructions-section ol {
            padding-right: 18px;
            margin-bottom: 0;
            list-style-type: disc;
            list-style-position: inside;
        }

        .instructions-section li {
            margin-bottom: 3px;
        }

        /* ------------------------------------------- */
        /* 5. 🖨️ إعدادات الطباعة للرولات (كل ملصق صفحة) */
        /* ------------------------------------------- */

@media print {
            /* 🛑 إعداد أبعاد الصفحة لتعادل أبعاد الملصق تماماً (140mm x 70mm) */
            @page {
                size: 140mm 70mm; /* العرض 140mm والارتفاع 70mm */
                margin: 0; /* إزالة هوامش الصفحة بالكامل */
            }

            body { 
                /* 🆕 إضافة هامش داخلي 1mm للصفحة المطبوعة */
                padding: 1mm !important; 
                margin: 0;
                background-color: #fff !important;
                /* تحديد عرض الجسم ليتناسب مع الهامش الجديد */
                width: 140mm;
                height: 70mm;
            }
            
            /* إخفاء جميع عناصر التحكم وواجهة المستخدم */
            .no-print, .pagination, form {
                display: none !important;
            }
            
            /* الملصق يصبح هو العنصر الوحيد على الصفحة */
            .barcode-container {
                display: flex; 
                /* 🆕 تصغير الحجم قليلاً ليتناسب مع هامش 1mm من كل جانب */
                width: 136mm; 
                height: 66mm; 
                
                margin:2mm auto; 
                /* إعادة إظهار الإطار الخارجي للملصق */
                border: 1px solid #000 !important; 
                box-shadow: none;
                border-radius: 0; 
                
                page-break-inside: avoid; 
                page-break-after: always; 
                overflow: hidden; 
            }
            
            /* آخر ملصق لا يحتاج لفاصل صفحة بعده */
            .barcode-container:last-child {
                page-break-after: auto;
            }

            /* 🖼️ قسم الباركود عند الطباعة */
            .barcode-section {
                width: 70mm; 
                height: 100%; 
                display: flex;
                flex-direction: column;
                justify-content: center;
                align-items: center;
                /* 🆕 إعادة الشريط الفاصل (بلون أغمق قليلاً للطباعة) */
                border-left: 1px dashed #666 !important; /* لون أغمق لضمان وضوحه */
                padding: 2mm;
                background-color: #fefefe !important; /* خلفية بيضاء فاتحة للقسم */
            }

            .barcode-section svg {
                width: 90%; 
                height: auto; 
                display: block;
                margin: 0;
                padding: 0;
            }
            
            .barcode-text {
                font-size: 16px;
                margin-top: 3px;
                font-weight: 700;
                color: #333 !important; /* ضمان اللون الأسود */
                letter-spacing: 1px;
            }

            /* 📝 قسم التعليمات عند الطباعة */
            .instructions-section {
                width: 70mm; 
                height: 100%; 
                display: flex;
                flex-direction: column;
                justify-content: center;
                text-align: right;
                /* 🆕 إعادة خلفية بلون فاتح للطباعة */
                background-color: #f7f7f7 !important; 
                color: #555 !important; /* لون نص فاتح قليلاً للقراءة */
                font-size: 10px; /* ضمان حجم خط واضح */
                line-height: 1.3;
                /* إعادة البادئة الداخلية */
                padding: 2mm 6mm 2mm 0.2mm !important; 
            }

            .instructions-section h6 {
                font-size: 12px;
                font-weight: 700;
                color: #007bff !important; /* 🆕 إعادة اللون الأزرق للعناوين */
                margin-bottom: 5px;
                border-bottom: 1px solid #ddd !important; /* 🆕 إعادة الحدود الفاصلة */
                padding-bottom: 2px;
            }
            
            .instructions-section ol {
                padding-right: 18px !important; 
                margin-bottom: 0 !important;
                list-style-type: disc !important;
                list-style-position: inside !important;
                color: #555 !important; /* تأكيد لون نص القائمة */
            }
            
            .instructions-section li {
                margin-bottom: 3px !important;
            }
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        
        <div class="no-print mb-4">
            <div class="card bg-primary text-white shadow-sm mb-3" style="margin-top: -15px;">
                <div class="card-body p-3 text-center d-flex justify-content-between align-items-center">
                    <h5 class="mb-0 fw-bold">
                        📄 أداة توليد وطباعة الباركود المؤقتة
                    </h5>
                    <span class="badge bg-warning text-dark">الصفحة: <?= $page ?> من <?= $total_pages ?></span>
                </div>
            </div>
            
            <div class="card shadow-sm p-3 mb-3">
                <form class="row g-3 align-items-end" method="get">
                    
                    <div class="col-auto">
                        <label for="prefix_select" class="form-label small fw-bold text-danger">🏷️ بادئة الباركود</label>
                        <select id="prefix_select" name="prefix" class="form-select form-select-sm" required>
                            <?php foreach ($available_prefixes as $p): ?>
                                <option value="<?= $p ?>" <?= ($p === $selected_prefix) ? 'selected' : '' ?>>
                                    <?= $p ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    
                    <div class="col-auto flex-grow-1">
                        <label for="start" class="form-label small fw-bold text-primary">🔢 توليد باركودات جديدة ابتداء من الرقم:</label>
                        <input type="number" id="start" name="start" value="<?= htmlspecialchars($start_number_input) ?>" 
                               class="form-control form-control-sm" min="1" required>
                    </div>
                    
                    <div class="col-auto">
                        <label for="print_count" class="form-label small fw-bold text-success">📄 عدد الملصقات (×100):</label>
                        <input type="number" id="print_count" name="print_count" value="<?= htmlspecialchars($print_pages_count) ?>" 
                               class="form-control form-control-sm" style="width: 120px;" min="1" max="100" required>
                    </div>
                    
                    <div class="col-auto">
                        <input type="hidden" name="p" value="<?= $page ?>"> 
                        <button type="submit" class="btn btn-warning btn-sm" style="height: 31px;">↩️ تحديث العرض</button>
                    </div>
                </form>
            </div>
            
            <div class="d-flex justify-content-between align-items-center mb-3 flex-wrap">
                <div>
                    <a href="/barcode-system/admin_dashboard.php" class="btn btn-info text-white me-2">العودة للوحة الإدارة</a>
                    <button onclick="window.print()" class="btn btn-success me-2">🖨️ طباعة الملصقات المعروضة (<?= count($barcodes) ?>)</button>
                </div>
                <div class="text-start mt-2 mt-md-0">
                    <p class="mb-1 small text-muted">⚠️ البادئة الحالية: **<?= $selected_prefix . date('ymd') ?>**</p>
                    <p class="mb-0 small text-danger fw-bold">تحذير: اختيار عدد كبير من الصفحات قد يبطئ المتصفح.</p>
                </div>
            </div>
            
        </div>

        <div class="d-flex flex-wrap justify-content-start align-items-center">
            <?php foreach ($barcodes as $barcode): ?>
                <div class="barcode-container">
                    
                    <div class="barcode-section">
                        <svg id="barcode-<?= htmlspecialchars($barcode) ?>"></svg>
                        <div class="barcode-text"><?= htmlspecialchars($barcode) ?></div>
                    </div>
                    
                    <div class="instructions-section">
                        <h6>دليل تعليمات استخدام المرتبة:</h6>
                        <ol>
                            <li>يجب عدم طوي المرتبة.</li>
                            <li>وضع المرتبة على قاعدة مسطحة ومستقيمة كي لا تتأثر جودة المرتبة.</li>
                            <li>يجب تدوير المرتبة بشكل دوري كل ثلاثة شهور للمحافظة على جودة المرتبة.</li>
                            <li>لا تدع الأطفال يستخدمونها للقفز أو اللعب.</li>
                            <li>لا تستخدم المقابض لرفع المرتبة فقد صنعت لمساعدتكم في تدوير المرتبة.</li>
                            <li>استعمال واقي المراتب يحميها من الأوساخ والغبار.</li>
                            <li>ولإزالة البقع عن المرتبة امسحها بقليل من الماء البارد والصابون وعدم استخدام المواد الكيمائية.</li>
                        </ol>
                    </div>

                </div>
            <?php endforeach; ?>
        </div>
        
        <nav class="no-print mt-5">
            <ul class="pagination justify-content-center shadow-sm rounded">
                <?php 
                // تحديث متغير $start_param ليشمل البادئة
                $start_param = '&start=' . $start_number_input . '&print_count=' . $print_pages_count . '&prefix=' . $selected_prefix; 
                ?>
                
                <?php if ($page > 1): ?>
                    <li class="page-item"><a class="page-link" href="?p=1<?= $start_param ?>">الأولى</a></li>
                <?php endif; ?>
                
                <?php if ($page > 1): ?>
                    <li class="page-item"><a class="page-link" href="?p=<?= $page - 1 ?><?= $start_param ?>">السابق</a></li>
                <?php endif; ?>
                
                <?php 
                $start = max(1, $page - 2);
                $end = min($total_pages, $page + 2);

                for ($i = $start; $i <= $end; $i++): 
                ?>
                    <li class="page-item <?= ($i == $page) ? 'active bg-primary' : '' ?>">
                        <a class="page-link" href="?p=<?= $i ?><?= $start_param ?>"><?= $i ?></a>
                    </li>
                <?php endfor; ?>
                
                <?php if ($page < $total_pages): ?>
                    <li class="page-item"><a class="page-link" href="?p=<?= $page + 1 ?><?= $start_param ?>">التالي</a></li>
                <?php endif; ?>

                <?php if ($page < $total_pages): ?>
                    <li class="page-item"><a class="page-link" href="?p=<?= $total_pages ?><?= $start_param ?>">الأخيرة</a></li>
                <?php endif; ?>
            </ul>
        </nav>
        
        <form class="no-print d-flex justify-content-center align-items-center mt-3" method="get">
            <label for="jump_page" class="form-label mb-0 me-2 small text-muted">انتقال لصفحة محددة:</label>
            <input type="number" id="jump_page" name="p" min="1" max="<?= $total_pages ?>" 
                   class="form-control form-control-sm me-2" style="width: 80px;" placeholder="الصفحة" required>
            <button type="submit" class="btn btn-secondary btn-sm">انتقال</button>
            <input type="hidden" name="start" value="<?= htmlspecialchars($start_number_input) ?>">
            <input type="hidden" name="print_count" value="<?= htmlspecialchars($print_pages_count) ?>">
            <input type="hidden" name="prefix" value="<?= htmlspecialchars($selected_prefix) ?>">
        </form>

    </div>

    <script>
        document.addEventListener("DOMContentLoaded", function() {
            <?php foreach ($barcodes as $barcode): ?>
                JsBarcode("#barcode-<?= htmlspecialchars($barcode) ?>", "<?= htmlspecialchars($barcode) ?>", {
                    format: "CODE128", 
                    displayValue: false, 
                    margin: 0, 
                    width: 1.5, /* زيادة عرض الخط قليلاً لجعله أوضح */
                    height: 50 
                });
            <?php endforeach; ?>
        });
    </script>
</body>
</html>