<?php
// FILE: admin/manage_users.php
require_once __DIR__ . '/../init.php';

// 🛑 التحقق من صلاحيات المدير العام فقط (Super Admin)
if (!isset($_SESSION['user']) || $_SESSION['user']['role'] !== 'super_admin') {
    // تصحيح التوجيه إلى لوحة التحكم الجديدة
    header('Location: /barcode-system/admin_dashboard.php'); 
    exit;
}

$current_user_id = $_SESSION['user']['id'];
$message = isset($_GET['msg']) ? htmlspecialchars($_GET['msg']) : '';
$message_type = isset($_GET['type']) ? htmlspecialchars($_GET['type']) : '';
$edit_id = $_GET['edit_id'] ?? null; // ID المستخدم المراد تعديله

// 1. معالجة طلب حذف المستخدم
if (isset($_GET['action']) && $_GET['action'] === 'delete' && isset($_GET['id'])) {
    $delete_id = (int)$_GET['id'];
    
    if ($delete_id === $current_user_id) {
        $message = 'لا يمكنك حذف حسابك الحالي.';
        $message_type = 'danger';
    } else {
        try {
            $stmt = $pdo->prepare('DELETE FROM users WHERE id = ?');
            $stmt->execute([$delete_id]);
            
            if ($stmt->rowCount()) {
                $message = 'تم حذف المستخدم بنجاح.';
                $message_type = 'success';
            } else {
                $message = 'فشل حذف المستخدم أو لم يتم العثور عليه.';
                $message_type = 'warning';
            }
        } catch (\PDOException $e) {
            $message = 'خطأ في قاعدة البيانات: ' . $e->getMessage();
            $message_type = 'danger';
        }
    }
    // إعادة التوجيه لمنع إعادة الإرسال بعد الحذف
    header("Location: " . $_SERVER['PHP_SELF'] . "?msg=" . urlencode($message) . "&type=" . $message_type);
    exit;
}

// 2. معالجة طلب تحديث المستخدم (POST)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['user_id_to_edit'])) {
    $target_user_id = $_POST['user_id_to_edit'];
    $new_username = trim($_POST['new_username'] ?? '');
    $new_password = $_POST['new_password'] ?? '';
    $new_role = $_POST['new_role'] ?? '';

    // التحقق من أن السوبر أدمن لا يغير دوره إلى شيء آخر
    if ($target_user_id == $current_user_id && $new_role !== 'super_admin') {
        $message = 'لا يمكنك تغيير دورك كسوبر أدمن من هنا.';
        $message_type = 'danger';
    }
    // التحقق من أن حقل الاسم ليس فارغاً
    elseif (empty($new_username)) {
        $message = 'اسم المستخدم الجديد مطلوب.';
        $message_type = 'danger';
    }
    // التأكد من طول كلمة المرور الجديدة
    elseif (!empty($new_password) && strlen($new_password) < 6) {
        $message = 'يجب أن لا تقل كلمة المرور الجديدة عن 6 أحرف.';
        $message_type = 'danger';
    } 
    else {
        $sql_parts = [];
        $params = [];
        
        $sql_parts[] = 'username = ?';
        $params[] = $new_username;

        $sql_parts[] = 'role = ?';
        $params[] = $new_role;

        // تحديث كلمة المرور (دون الحاجة للقديمة)
        if (!empty($new_password)) {
            $new_password_hash = password_hash($new_password, PASSWORD_DEFAULT);
            $sql_parts[] = 'password_hash = ?';
            $params[] = $new_password_hash;
        }
        
        $sql = 'UPDATE users SET ' . implode(', ', $sql_parts) . ' WHERE id = ?';
        $params[] = $target_user_id;

        try {
            $update_stmt = $pdo->prepare($sql);
            $update_stmt->execute($params);

            $message = 'تم تحديث بيانات المستخدم بنجاح. ' . (empty($new_password) ? '' : 'تم تغيير كلمة المرور أيضاً.');
            $message_type = 'success';
            // تحديث الجلسة إذا كان المستخدم هو السوبر أدمن نفسه
            if ($target_user_id == $current_user_id) {
                $_SESSION['user']['username'] = $new_username;
                $_SESSION['user']['role'] = $new_role;
            }
        } catch (PDOException $e) {
            $message = 'خطأ في قاعدة البيانات: قد يكون اسم المستخدم مكرراً.';
            $message_type = 'danger';
        }
    }

    // إعادة التوجيه لمنع إعادة الإرسال بعد التحديث
    header("Location: " . $_SERVER['PHP_SELF'] . "?msg=" . urlencode($message) . "&type=" . $message_type);
    exit;
}

// 🗂️ جلب جميع المستخدمين
$users_query = 'SELECT id, username, role, created_at FROM users ORDER BY id DESC';
$users = $pdo->query($users_query);
$user_list = $users->fetchAll(PDO::FETCH_ASSOC);

$edit_user = null;
if ($edit_id) {
    $edit_stmt = $pdo->prepare('SELECT id, username, role FROM users WHERE id = ?');
    $edit_stmt->execute([$edit_id]);
    $edit_user = $edit_stmt->fetch(PDO::FETCH_ASSOC);
}
?>
<!doctype html>
<html lang="ar">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width,initial-scale=1">
    <title>إدارة المستخدمين</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body{font-family:Tajawal,sans-serif;background:#f6f8fa;padding:16px;direction:rtl;}
        .card{border-radius:12px;padding:16px;margin-bottom:16px;box-shadow:0 4px 12px rgba(0,0,0,0.05);}
    </style>
</head>
<body>
    <div class="container">
        <h4 class="mb-4">إدارة المستخدمين</h4>

        <?php if ($message): ?>
            <div class="alert alert-<?= $message_type ?>"><?= $message ?></div>
        <?php endif; ?>

        <div class="mb-3 d-flex justify-content-between">
            <a href="add_user.php" class="btn btn-primary">➕ إضافة مستخدم جديد</a>
            <a href="/barcode-system/admin_dashboard.php" class="btn btn-secondary">العودة للوحة التحكم</a>
        </div>

        <?php if ($edit_user): ?>
            <div class="card mb-4">
                <div class="card-header bg-warning text-dark">
                    تعديل المستخدم: **<?php echo htmlspecialchars($edit_user['username']); ?>**
                </div>
                <div class="card-body">
                    <form method="post" action="manage_users.php">
                        <input type="hidden" name="user_id_to_edit" value="<?php echo $edit_user['id']; ?>">
                        
                        <div class="mb-3">
                            <label for="new_username" class="form-label">اسم المستخدم الجديد</label>
                            <input type="text" name="new_username" id="new_username" class="form-control" value="<?php echo htmlspecialchars($edit_user['username']); ?>" required>
                        </div>
                        
                        <div class="mb-3">
                            <label for="new_password" class="form-label">كلمة المرور الجديدة (اختياري)</label>
                            <input type="password" name="new_password" id="new_password" class="form-control" placeholder="اترك فارغاً لعدم التغيير">
                            <small class="form-text text-muted">لتغيير كلمة المرور دون القديمة، أدخل كلمة جديدة لا تقل عن 6 أحرف.</small>
                        </div>

                        <div class="mb-3">
                            <label for="new_role" class="form-label">الدور</label>
                            <select name="new_role" id="new_role" class="form-select" required>
                                <option value="super_admin" <?php echo ($edit_user['role'] === 'super_admin' ? 'selected' : ''); ?>>مدير عام (Super Admin)</option>
                                <option value="admin" <?php echo ($edit_user['role'] === 'admin' ? 'selected' : ''); ?>>مدير مقيد (Admin)</option>
                                <option value="delivery_agent" <?php echo ($edit_user['role'] === 'delivery_agent' ? 'selected' : ''); ?>>مندوب تسليم</option>
                            </select>
                            <?php if ($edit_user['id'] == $current_user_id): ?>
                                <small class="form-text text-danger">لا يمكنك تغيير دورك من سوبر أدمن إلى دور آخر.</small>
                            <?php endif; ?>
                        </div>
                        
                        <button type="submit" class="btn btn-success">حفظ التعديلات</button>
                        <a href="manage_users.php" class="btn btn-secondary">إلغاء</a>
                    </form>
                </div>
            </div>
        <?php endif; ?>


        <div class="card">
            <h5>قائمة المستخدمين</h5>
            <div class="table-responsive">
                <table class="table table-striped table-hover table-sm">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>اسم المستخدم</th>
                            <th>الصلاحية</th>
                            <th>تاريخ الإنشاء</th>
                            <th>إجراءات</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach($user_list as $u): ?>
                        <tr>
                            <td><?= $u['id'] ?></td>
                            <td>
                                <?= htmlspecialchars($u['username']) ?>
                                <?php if ($u['id'] == $current_user_id): ?>
                                    <span class="badge text-bg-info">أنت</span>
                                <?php endif; ?>
                            </td>
                            
                            <td>
                                <?php
                                switch($u['role']) {
                                    case 'super_admin': echo '<span class="badge text-bg-danger">مدير عام</span>'; break;
                                    case 'admin': echo '<span class="badge text-bg-warning">مدير مقيد</span>'; break;
                                    case 'delivery_agent': echo '<span class="badge text-bg-success">مندوب تسليم</span>'; break; 
                                    default: echo '<span class="badge text-bg-secondary">' . htmlspecialchars($u['role']) . '</span>';
                                }
                                ?>
                            </td>
                            
                            <td><?= date('Y-m-d', strtotime($u['created_at'])) ?></td>
                            
                            <td>
                                <a href="?edit_id=<?= $u['id'] ?>" class="btn btn-sm btn-info text-white">تعديل</a>
                                <?php if ($u['id'] !== $current_user_id): ?>
                                    <button onclick="deleteUser(<?= $u['id'] ?>,'<?= htmlspecialchars($u['username']) ?>')" class="btn btn-sm btn-danger">حذف</button>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    
    <script>
        function deleteUser(id, username) {
            if (confirm("هل أنت متأكد من حذف المستخدم '" + username + "'؟")) {
                window.location.href = `manage_users.php?action=delete&id=${id}`;
            }
        }
    </script>
</body>
</html>