<?php
require_once __DIR__ . '/../init.php';
header('Content-Type: application/json; charset=utf-8');

$method = $_SERVER['REQUEST_METHOD'];

function cleanBarcode($barcode){
    // إزالة رموز التحكم والمسافات البادئة/اللاحقة
    return preg_replace('/[\x00-\x1F\x7F]+/', '', trim($barcode));
}

if($method === 'POST'){
    $data = json_decode(file_get_contents('php://input'), true) ?: $_POST;

    // 🔍 البحث بالباركود فقط (الاستعلام الخاص باليوزر العادي)
    if(isset($data['barcode']) && !isset($data['name'])){
        $barcode = cleanBarcode($data['barcode']);
        
        try {
            // تحديد الأعمدة صراحةً
            $stmt = $pdo->prepare("SELECT id, barcode, name, size, production_date, notes, delivery_note_ref, delivery_date FROM products WHERE barcode = ?");
            
            $stmt->execute([$barcode]);
            $product = $stmt->fetch(PDO::FETCH_ASSOC);
            
            // إرجاع المنتج أو رسالة الخطأ المخصصة في حال عدم العثور عليه
            echo json_encode($product ?: ['error'=>'لا يوجد منتج مرتبط بهذا الباركود']);
            
        } catch(\PDOException $e) {
            http_response_code(500); 
            echo json_encode(['error'=>'فشل الاتصال بقاعدة البيانات أو تنفيذ الاستعلام: ' . $e->getMessage()]);
            
        } catch(Exception $e) {
            http_response_code(500); 
            echo json_encode(['error'=>'خطأ غير متوقع أثناء البحث: ' . $e->getMessage()]);
        }
        exit;
    }

    // ➕ إضافة منتج جديد (POST - تم حذف expiry_date)
    $barcode = cleanBarcode($data['barcode'] ?? '');
    $name = trim($data['name'] ?? '');
    $size = trim($data['size'] ?? '');
    $production_date = $data['production_date'] ?: null; // تاريخ الإنتاج الآلي
    $notes = $data['notes'] ?: null; // تم حذف expiry_date

    if(!$barcode || !$name){
        http_response_code(400);
        echo json_encode(['error'=>'الباركود واسم المنتج مطلوبان']);
        exit;
    }

    $check = $pdo->prepare("SELECT id FROM products WHERE barcode = ?");
    $check->execute([$barcode]);
    if($check->fetch()){
        http_response_code(409);
        echo json_encode(['error'=>'رمز الباركود مسجل مسبقاً']);
        exit;
    }

    try {
        // تم حذف expiry_date من الاستعلام
        $stmt = $pdo->prepare("INSERT INTO products (barcode, name, size, production_date, notes, created_by) VALUES (?, ?, ?, ?, ?, ?)");
        $stmt->execute([$barcode, $name, $size ?: null, $production_date, $notes, $_SESSION['user']['id'] ?? null]);
        echo json_encode(['ok'=>true, 'id'=>$pdo->lastInsertId()]);
    } catch(Exception $e){
        http_response_code(500);
        echo json_encode(['error'=>'حدث خطأ أثناء الحفظ: '.$e->getMessage()]);
    }
    exit;
}

if($method === 'GET'){
    $q = $_GET['q'] ?? '';
    if($q){
        // تحديد الأعمدة صراحةً (تم حذف expiry_date)
        $stmt = $pdo->prepare("SELECT id, barcode, name, size, production_date, notes, delivery_note_ref, delivery_date FROM products WHERE barcode LIKE ? OR name LIKE ? ORDER BY created_at DESC LIMIT 500");
        $stmt->execute(["%$q%", "%$q%"]);
    } else {
        // تحديد الأعمدة صراحةً (تم حذف expiry_date)
        $stmt = $pdo->query("SELECT id, barcode, name, size, production_date, notes, delivery_note_ref, delivery_date FROM products ORDER BY created_at DESC LIMIT 500");
    }
    echo json_encode($stmt->fetchAll());
    exit;
}

// FILE: api/products.php - كتلة PUT المُعدلة (تم حذف expiry_date)
if($method === 'PUT'){
    // 🛑 تقييد التعديل على Super Admin فقط
    $user_role = $_SESSION['user']['role'] ?? null;
    $is_super_admin = ($user_role === 'super_admin');
    $current_user_id = $_SESSION['user']['id'] ?? null;

    if (!$is_super_admin) {
        // لا يوجد صلاحية تعديل عامة لغير Super Admin
        http_response_code(403);
        echo json_encode(['error'=>'غير مصرح لك بتعديل المنتجات.']);
        exit;
    }

    parse_str(file_get_contents('php://input'), $put);
    $id = $put['id'] ?? null;
    
    // تأمين أساسي: يجب وجود ID
    if(!$id){
        http_response_code(400); echo json_encode(['error'=>'id required']); exit;
    }

    try{
        // 1. استلام الحقول الجديدة
        $barcode = cleanBarcode($put['barcode'] ?? '');
        $name = trim($put['name'] ?? '');
        $size = trim($put['size'] ?? '');
        $production_date = $put['production_date'] ?: null; 
        $notes = $put['notes'] ?: null; 

        $delivery_note_ref = $put['delivery_note_ref'] ?? null;
        $delivery_date = $put['delivery_date'] ?: null;

        // 🛑 منطق تحديث delivery_recorded_by 🛑
        // إذا تم إرسال أي من حقول التسليم (رقم السند أو التاريخ)، يجب تحديث المسجل إلى السوبر أدمن الحالي.
        $delivery_recorded_by = $put['delivery_recorded_by'] ?? null; 

        if (
            (isset($put['delivery_note_ref']) && $put['delivery_note_ref'] !== '') || 
            (isset($put['delivery_date']) && $put['delivery_date'] !== '')
        ) {
             // بما أن التعديل الآن مقصور على السوبر أدمن، فإننا نسجل هويته عند تعديل بيانات التسليم.
            $delivery_recorded_by = $current_user_id; 
        }

        // 🎯 3. تنفيذ استعلام التحديث
        $stmt = $pdo->prepare("
            UPDATE products SET 
                barcode=?, name=?, production_date=?, notes=?, size=?, 
                delivery_note_ref=?, delivery_date=?, delivery_recorded_by=? 
            WHERE id=?
        ");
        
        $stmt->execute([
            $barcode,
            $name,
            $production_date,
            $notes,
            $size,
            $delivery_note_ref,
            $delivery_date,
            $delivery_recorded_by, 
            $id
        ]);
        
        echo json_encode(['ok'=>true]);
    }catch(Exception $e){
        http_response_code(500); echo json_encode(['error'=>$e->getMessage()]);
    }
    exit;
}

if($method === 'DELETE'){
    // 🛑 تقييد الحذف على Super Admin فقط
    if (!isset($_SESSION['user']) || $_SESSION['user']['role'] !== 'super_admin') {
        http_response_code(403); 
        echo json_encode(['error'=>'غير مصرح لك بحذف المنتجات.']); 
        exit;
    }
    parse_str(file_get_contents('php://input'), $del);
    $id = $del['id'] ?? null;
    if(!$id){
        http_response_code(400); echo json_encode(['error'=>'id required']); exit;
    }
    try{
        $stmt = $pdo->prepare("DELETE FROM products WHERE id=?");
        $stmt->execute([$id]);
        echo json_encode(['ok'=>true]);
    }catch(Exception $e){
        http_response_code(500); echo json_encode(['error'=>$e->getMessage()]);
    }
    exit;
}

http_response_code(405);
echo json_encode(['error'=>'Method not allowed']);
?>