<?php
// FILE: admin/manage_qr_cache.php (النسخة النهائية والمُحسَّنة)

require_once __DIR__ . '/../init.php';

// 🛑 التحقق من صلاحية السوبر أدمن فقط
if (!isset($_SESSION['user']) || $_SESSION['user']['role'] !== 'super_admin') {
    header('Location: ' . $config['base_url'] . 'login.php');
    exit;
}

$message = '';
$error = '';

// **********************************
// 1. معالجة طلب التعديل (حذف الكاش + تسجيل التسمية الجديدة)
// **********************************
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'update_name') {
    $url_to_update = filter_var($_POST['url_to_update'] ?? '', FILTER_SANITIZE_URL);
    $new_product_name = trim($_POST['new_product_name'] ?? '');

    if (empty($url_to_update)) {
        $error = 'الرابط غير صحيح.';
    } else {
        try {
            $checksum = md5($url_to_update);
            
            // 1. حذف الإدخال من جدول الكاش (لإجبار النظام على إعادة التوليد)
            $pdo->prepare("DELETE FROM qr_codes WHERE checksum = ?")->execute([$checksum]);

            // 2. 🟢 إضافة إدخال جديد ببيانات فارغة والتسمية الجديدة (لتسجيل التسمية المحدثة فوراً)
            // نستخدم NOW() لتحديث created_at ليكون آخر وقت تحديث
            $stmt = $pdo->prepare("INSERT INTO qr_codes (url_target, product_name, checksum, created_at) VALUES (?, ?, ?, NOW())");
            $stmt->bindValue(1, $url_to_update);
            $stmt->bindValue(2, $new_product_name);
            $stmt->bindValue(3, $checksum);
            $stmt->execute();

            if ($stmt->rowCount() > 0) {
                 $message = "✅ تم تحديث التسمية بنجاح إلى: **" . htmlspecialchars($new_product_name) . "**، وتم مسح الكاش. <br> **الإجراء التالي:** يرجى إعادة زيارة رابط توليد QR لهذا المنتج لضمان حفظ الصورة الجديدة.";
            } else {
                 $error = "⚠️ لم يتم العثور على رمز QR مسجل لهذا الرابط في التخزين المؤقت، لكن تم تسجيل التسمية الجديدة **" . htmlspecialchars($new_product_name) . "**.";
            }
            
        } catch (PDOException $e) {
            $error = "❌ خطأ في قاعدة البيانات: " . $e->getMessage();
        }
    }
}

// **********************************
// 2. جلب جميع الروابط المخزنة (للعرض)
// **********************************
$qr_list = [];
try {
    // 🟢 جلب عمود product_name الجديد
    $stmt = $pdo->query("SELECT url_target, checksum, product_name, created_at FROM qr_codes GROUP BY checksum ORDER BY created_at DESC");
    $qr_list = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $error = "❌ فشل في جلب قائمة رموز QR المخزنة: " . $e->getMessage();
}

$page_title = 'إدارة وتعديل رموز QR';

// 🔴 دالة لتحديد فئة اللون بناءً على تاريخ الإنشاء
function getRowClass($createdAt) {
    $now = time();
    $createdTimestamp = strtotime($createdAt);
    $diffDays = ($now - $createdTimestamp) / (60 * 60 * 24);

    if ($diffDays <= 7) {
        // حديث (أقل من أسبوع)
        return 'table-success'; 
    } elseif ($diffDays >= 30) {
        // قديم (أكثر من شهر)
        return 'table-warning'; 
    }
    // لا يوجد تلوين خاص
    return ''; 
}
?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width,initial-scale=1">
    <title><?php echo $page_title; ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.5/font/bootstrap-icons.css">
    <style>
        /* 🔴 تعريف اللون الأساسي للعلامة التجارية لـ BackCare */
        :root {
            --backcare-primary: #0056b3; 
            --backcare-accent: #007bff;
        }

        body { direction: rtl; background-color: #f0f2f5; } /* خلفية رمادية فاتحة جداً */
        .main-card { 
            border: none;
            border-radius: 1rem;
            box-shadow: 0 0.5rem 1rem rgba(0,0,0,0.1);
            padding: 20px;
        }
        .table-container {
            max-height: 65vh; 
            overflow-y: auto; 
        }

        /* 🔴 تحديد حدود الخلايا باللون الأبيض */
        .table-bordered > :not(caption) > * > * {
            border: 1px solid #fff; /* حدود بيضاء للخلايا */
        }

        /* تحديد عرض الأعمدة بالنسبة المئوية */
        .url-cell { width: 35%; max-width: 350px; overflow: hidden; text-overflow: ellipsis; white-space: nowrap; }
        .label-cell { width: 25%; }
        .checksum-cell { width: 15%; }
        .date-cell { width: 15%; }
        .action-cell { width: 10%; min-width: 120px; } /* ضمان عرض كافٍ للزر */

        .checksum-text { font-size: 0.75rem; color: #6c757d; }
        .badge-product { font-size: 0.9rem; padding: 0.5em 0.7em; }
        
        .table thead th { 
            background-color: var(--backcare-primary); 
            color: #fff; 
            border-color: #fff; /* التأكد من أن حدود رأس الجدول بيضاء أيضاً */
        }
        .container { margin-top: 30px; margin-bottom: 30px; }
        
        /* 📱 تحسين التوافق مع الجوال */
        @media (max-width: 768px) {
            .table-container { border: none; }
            .table tbody td, .table thead th { font-size: 0.8rem; }
            
            /* تعديل عرض الجوال: إعطاء الأولوية للرابط والتسمية */
            .url-cell { width: 40%; max-width: none; }
            .label-cell { width: 30%; }
            .date-cell { width: 20%; }
            .action-cell { width: 10%; min-width: 90px; } 
        }
    </style>
</head>
<body>
    <div class="container">
        <a href="../admin_dashboard.php" class="btn btn-secondary btn-sm mb-4">
            <i class="bi bi-arrow-right-short me-2"></i> العودة للوحة التحكم
        </a>
        <h1 class="mb-4 text-primary"><i class="bi bi-qr-code-scan me-2"></i> <?php echo $page_title; ?></h1>

        <div class="card main-card">
            <div class="card-body">

                <?php if ($message): ?>
                    <div class="alert alert-success alert-dismissible fade show fw-bold" role="alert">
                        <?php echo $message; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                <?php endif; ?>
                <?php if ($error): ?>
                    <div class="alert alert-danger alert-dismissible fade show fw-bold" role="alert">
                        <?php echo $error; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                    </div>
                <?php endif; ?>
                
                <p class="text-muted mb-4">قائمة بالـ QR Codes المخزنة بالنظام . مع إمكانية الحذف والتعديل لتغيير أسماء المنتجات.</p>
                
                <h4 class="mb-3 text-secondary"><i class="bi bi-table me-2"></i> سجلات الكاش</h4>
                <div class="table-responsive table-container">
                    <table class="table table-striped table-hover table-sm table-bordered">
                        <thead>
                            <tr>
                                <th class="url-cell">الرابط المستهدف (URL)</th>
                                <th class="label-cell">التسمية الحالية</th> 
                                <th class="checksum-cell d-none d-md-table-cell">رمز التحقق</th>
                                <th class="date-cell">آخر تحديث</th>
                                <th class="action-cell">الإجراء</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($qr_list)): ?>
                                <tr>
                                    <td colspan="5" class="text-center py-4">لا توجد رموز QR مخزنة حالياً.</td>
                                </tr>
                            <?php else: ?>
                                <?php foreach ($qr_list as $qr): ?>
                                    <?php 
                                        // تطبيق فئة اللون على الصف
                                        $rowClass = getRowClass($qr['created_at']);
                                    ?>
                                    <tr class="<?php echo $rowClass; ?>">
                                        <td class="url-cell">
                                            <a href="<?php echo htmlspecialchars($qr['url_target']); ?>" target="_blank" title="<?php echo htmlspecialchars($qr['url_target']); ?>">
                                                <?php echo htmlspecialchars($qr['url_target']); ?>
                                            </a>
                                        </td>
                                        <td class="label-cell">
                                            <span class="badge rounded-pill bg-dark text-light badge-product">
                                                <?php echo htmlspecialchars($qr['product_name'] ?? 'لا يوجد اسم'); ?>
                                            </span>
                                        </td>
                                        <td class="checksum-cell d-none d-md-table-cell">
                                            <span class="checksum-text"><?php echo $qr['checksum']; ?></span>
                                        </td>
                                        <td class="date-cell">
                                            <small class="text-secondary"><?php echo date('Y-m-d H:i', strtotime($qr['created_at'])); ?></small>
                                        </td>
                                        <td class="action-cell">
                                            <button class="btn btn-sm btn-warning w-100" 
                                                    data-bs-toggle="modal" 
                                                    data-bs-target="#editModal" 
                                                    data-url="<?php echo htmlspecialchars($qr['url_target']); ?>"
                                                    data-name="<?php echo htmlspecialchars($qr['product_name'] ?? ''); ?>">
                                                <i class="bi bi-pencil-square"></i> تعديل
                                            </button>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
        </div>

    <div class="modal fade" id="editModal" tabindex="-1" aria-labelledby="editModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-dialog-centered"> 
            <div class="modal-content">
                <form method="POST" action="">
                    <input type="hidden" name="action" value="update_name">
                    <div class="modal-header bg-primary text-white"> 
                        <h5 class="modal-title" id="editModalLabel"><i class="bi bi-pencil-square me-2"></i> تعديل تسمية المنتج</h5>
                        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
                    </div>
                    
                    <div class="modal-body">
                        
                        <div class="alert alert-warning border-0 p-3 mb-3" role="alert">
                            <i class="bi bi-exclamation-triangle-fill me-2"></i>
                            **** سيتم **تحديث الاسم** ومسح **الاسم القديم** لهذا الرمز. يجب إعادة توليد الرمز مرة أخرى بعد الإغلاق.
                        </div>

                        <div class="mb-3">
                            <label for="modalUrl" class="form-label float-end fw-bold text-muted">الرابط المستهدف (للعرض فقط)</label>
                            <input type="url" class="form-control bg-light text-primary border-primary border-2" id="modalUrl" name="url_to_update" readonly required>
                        </div>
                        
                        <div class="mb-3">
                            <label for="newProductName" class="form-label float-end fw-bold">التسمية الجديدة (اسم المنتج)</label>
                            <input type="text" class="form-control" id="newProductName" name="new_product_name" placeholder="أدخل اسم المنتج الجديد هنا" required>
                            <div class="form-text">سيتم استخدام هذه التسمية في قاعدة البيانات وأسفل رمز QR الجديد.</div>
                        </div>
                    </div>
                    
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal"><i class="bi bi-x-circle-fill"></i> إلغاء</button>
                        <button type="submit" class="btn btn-primary"><i class="bi bi-save me-1"></i> حفظ التسمية وتحديث الكاش</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // تمرير الـ URL والاسم لنموذج التعديل عند الضغط على الزر
        var editModal = document.getElementById('editModal');
        editModal.addEventListener('show.bs.modal', function (event) {
            var button = event.relatedTarget;
            var url = button.getAttribute('data-url');
            var name = button.getAttribute('data-name'); 
            
            var modalUrlInput = editModal.querySelector('#modalUrl');
            var modalNameInput = editModal.querySelector('#newProductName');
            
            modalUrlInput.value = url;
            modalNameInput.value = name; 
        });
    </script>
</body>
</html>