<?php
// FILE: admin/add_user.php
require_once __DIR__ . '/../init.php';

// التحقق من صلاحيات المدير العام فقط (Super Admin)
if (!isset($_SESSION['user']) || $_SESSION['user']['role'] !== 'super_admin') {
    // التوجيه الآمن باستخدام المسار الكامل
    header('Location: /' . $config['base_url'] . 'admin_dashboard.php?error=unauthorized');
    exit;
}

$message = '';
$message_type = '';

// --------------------------------------------------------------------
// 1. تعريف الصلاحيات المتاحة (المحدثة لتشمل كل روابط اللوحة)
// --------------------------------------------------------------------
$all_permissions = [
    'products' => [
        'label' => 'إدارة المنتجات',
        'perms' => [
            'products_view'         => '1. عرض قائمة المنتجات',
            'products_add'          => '2. إضافة منتجات جديدة',
            'products_export'       => '3. تصدير قائمة المنتجات (CSV)',
            'products_delete'       => '4. حذف المنتجات',
        ]
    ],
    'barcode_qr' => [
        'label' => 'الباركود و QR Code',
        'perms' => [
            'barcode_generate_print'=> '1. توليد وطباعة الباركود',
            'barcode_heavy_print'   => '2. طباعة باركود (+100K)',
            'qr_manage'             => '3. إدارة (QR Code) المسجلة',
            'qr_generate_link'      => '4. مولد رمز QR للروابط',
        ]
    ],
    'delivery' => [
        'label' => 'إدارة التسليم والسندات',
        'perms' => [
            'delivery_note_manage'  => '1. إدارة سندات التسليم',
            'delivery_assign_product' => '2. تعيين منتج لسند تسليم',
        ]
    ],
    'user_management' => [
        'label' => 'إدارة المستخدمين',
        'perms' => [
            'users_view_manage'     => '1. عرض/إدارة المستخدمين',
        ]
    ],
    'reporting' => [
        'label' => 'التقارير والإعدادات',
        'perms' => [
            'reports_view'          => '1. عرض التقارير والإحصائيات',
        ]
    ]
];
// --------------------------------------------------------------------


// لجلب الأدوار المعرفة في init.php
$allowed_roles = array_keys($config['user_roles'] ?? []);
$username = '';
$role = 'admin'; // تحديد قيمة افتراضية

// لملء الحقول في حال فشل الإرسال
$selected_permissions = [];


if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = trim($_POST['username'] ?? '');
    $password = $_POST['password'] ?? '';
    $role = $_POST['role'] ?? 'user';
    // جلب الصلاحيات المختارة
    $selected_permissions = $_POST['permissions'] ?? [];

    // تحديد الصلاحيات التي سيتم حفظها في قاعدة البيانات (permissions_json)
    $permissions_json = null;
    if ($role === 'super_admin') {
        // السوبر أدمن لا يحتاج لصلاحيات مخصصة
        $permissions_json = null;
    } else {
        // تصفية الصلاحيات والتحويل إلى JSON: البدء بتجميع كل الصلاحيات الصالحة
        $valid_perms = [];
        foreach ($all_permissions as $group) {
            $valid_perms = array_merge($valid_perms, array_keys($group['perms']));
        }

        // تطبيق التصفية باستخدام Anonymous Function
        $final_permissions = array_filter($selected_permissions, function($p) use ($valid_perms) {
            return in_array($p, $valid_perms);
        });

        $permissions_json = json_encode($final_permissions);
    }

    if (empty($username) || empty($password)) {
        $message = 'الرجاء إدخال اسم المستخدم وكلمة المرور.';
        $message_type = 'danger';
    } elseif (strlen($password) < 6) {
        $message = 'يجب أن لا تقل كلمة المرور عن 6 أحرف.';
        $message_type = 'danger';
    } elseif (!in_array($role, $allowed_roles)) {
        $message = 'صلاحية المستخدم غير صالحة.';
        $message_type = 'danger';
    } else {
        $check_stmt = $pdo->prepare('SELECT id FROM users WHERE username = ?');
        $check_stmt->execute([$username]);

        if ($check_stmt->fetch()) {
            $message = 'اسم المستخدم موجود بالفعل. الرجاء اختيار اسم آخر.';
            $message_type = 'danger';
        } else {
            $password_hash = password_hash($password, PASSWORD_DEFAULT);

            try {
                // تم إضافة عمود permissions_json في الاستعلام
                $insert_stmt = $pdo->prepare('INSERT INTO users (username, password_hash, role, permissions_json) VALUES (?, ?, ?, ?)');
                $insert_stmt->execute([$username, $password_hash, $role, $permissions_json]);

                $role_display = $config['user_roles'][$role] ?? $role;

                $message = 'تم إنشاء المستخدم **' . htmlspecialchars($username) . '** بصلاحية **' . htmlspecialchars($role_display) . '** بنجاح.';
                $message_type = 'success';
                // إعادة تعيين المتغيرات بعد النجاح
                $username = '';
                $role = 'admin';
                $selected_permissions = [];
            } catch(\PDOException $e) {
                $message = 'حدث خطأ في قاعدة البيانات: ' . $e->getMessage();
                $message_type = 'danger';
            }
        }
    }
}
?>
<!doctype html>
<html lang="ar">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width,initial-scale=1">
<title>إضافة مستخدم جديد</title>
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
<style>
body{font-family:Tajawal,sans-serif;background:#f6f8fa;padding:16px;direction:rtl;}
.card{border-radius:12px;padding:30px;margin-bottom:16px;box-shadow:0 6px 18px rgba(0,0,0,0.1);}
.form-control, .form-select {border-radius: 8px;}
.btn {border-radius: 8px;}
.permissions-group { border: 1px solid #e0e0e0; border-radius: 8px; padding: 10px; margin-bottom: 15px; background-color: #fcfcfc; }
.permissions-group-header { font-weight: bold; margin-bottom: 10px; padding-bottom: 5px; border-bottom: 1px dashed #ddd; }
</style>
</head>
<body>
<div class="container">
<div class="card" style="max-width: 700px; margin: 50px auto;">
<h4 class="mb-4 text-center text-primary">إنشاء حساب جديد</h4>

<?php if ($message): ?>
<div class="alert alert-<?= $message_type ?>"><?= $message ?></div>
<?php endif; ?>

<form method="post">
<div class="row">
    <div class="col-md-6 mb-3">
        <label class="form-label">اسم المستخدم (Username)</label>
        <input type="text" name="username" class="form-control" required placeholder="أدخل اسم المستخدم" value="<?= htmlspecialchars($username) ?>">
    </div>
    <div class="col-md-6 mb-3">
        <label class="form-label">كلمة المرور</label>
        <input type="password" name="password" class="form-control" required minlength="6" placeholder="يجب أن لا تقل عن 6 أحرف">
    </div>
</div>

<div class="mb-4">
    <label class="form-label">الصلاحية</label>
    <select name="role" class="form-select" id="new_role">
    <?php foreach ($config['user_roles'] as $role_key => $role_name): ?>
        <option value="<?= $role_key ?>" <?php echo ($role_key === $role ? 'selected' : ''); ?>>
            <?= htmlspecialchars($role_name) ?>
        </option>
    <?php endforeach; ?>
    </select>
</div>

<div class="mb-4" id="permissionsContainer">
    <label class="form-label">الصلاحيات المخصصة (تطبق فقط إذا كان الدور **مدير مقيد** أو **مندوب تسليم**)</label>
    <div class="row">
        <?php foreach ($all_permissions as $group_key => $group): ?>
            <div class="col-md-4">
                <div class="permissions-group">
                    <div class="permissions-group-header"><?= htmlspecialchars($group['label']) ?></div>
                    <?php foreach ($group['perms'] as $perm_key => $perm_label): ?>
                        <div class="form-check">
                            <input class="form-check-input"
                                        type="checkbox"
                                        value="<?= htmlspecialchars($perm_key) ?>"
                                        id="<?= htmlspecialchars('add_' . $perm_key) ?>"
                                        name="permissions[]"
                                        <?php echo in_array($perm_key, $selected_permissions) ? 'checked' : ''; ?>
                                        >
                            <label class="form-check-label" for="<?= htmlspecialchars('add_' . $perm_key) ?>">
                                <?= htmlspecialchars($perm_label) ?>
                            </label>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        <?php endforeach; ?>
    </div>
</div>
<div class="d-grid gap-2">
<button type="submit" class="btn btn-primary btn-lg">إنشاء المستخدم</button>
<a href="manage_users.php" class="btn btn-outline-secondary">العودة لإدارة المستخدمين</a>
</div>
</form>
</div>
</div>

<script>
    // JavaScript لإظهار/إخفاء حاوية الصلاحيات بناءً على الدور
    const roleSelect = document.getElementById('new_role');
    const permissionsContainer = document.getElementById('permissionsContainer');

    function togglePermissions() {
        // الصلاحيات المخصصة تظهر فقط للأدوار غير "super_admin"
        if (roleSelect.value === 'super_admin') {
            permissionsContainer.style.display = 'none';
        } else {
            permissionsContainer.style.display = 'block';
        }
    }

    // استدعاء عند التحميل الأولي
    if (roleSelect) {
        togglePermissions();
        // استدعاء عند تغيير قيمة القائمة المنسدلة
        roleSelect.addEventListener('change', togglePermissions);
    }
</script>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>