<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Question;
use App\Models\Answer;
use App\Models\Lead; 
use App\Models\Setting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;

class QuestionController extends Controller
{
    /**
     * لوحة البيانات (Dashboard)
     * إصلاح خطأ المتغيرات الناقصة للرسم البياني
     */
    public function dashboard()
    {
        $questionsCount = Question::count();
        $leadsCount = Lead::count();
        $latestLeads = Lead::latest()->take(5)->get();

        $chartData = Lead::selectRaw('DATE(created_at) as date, COUNT(*) as count')
            ->where('created_at', '>=', now()->subDays(7))
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        $chartLabels = $chartData->pluck('date');
        $chartValues = $chartData->pluck('count');

        return view('admin.dashboard', compact('questionsCount', 'leadsCount', 'latestLeads', 'chartLabels', 'chartValues'));
    }

    /**
     * صفحة الإعدادات وفيديو الإنترو
     * إصلاح خطأ image_66ee81 وضمان الحفظ في image_669465
     */
    public function editSettings()
    {
        $settings = Setting::first();
        return view('admin.settings.index', compact('settings'));
    }

    public function updateSettings(Request $request)
    {
        $settings = Setting::first() ?: new Setting();
        $data = $request->only(['primary_color', 'bg_color', 'wa_enabled', 'wa_token', 'wa_instance_id', 'wa_api_url']);

        if ($request->hasFile('intro_video')) {
            if ($settings->intro_video_path) {
                Storage::disk('public')->delete($settings->intro_video_path);
            }
            $data['intro_video_path'] = $request->file('intro_video')->store('videos', 'public');
        }

        $settings->fill($data)->save();
        return back()->with('success', 'تم تحديث الإعدادات بنجاح');
    }

    // دالة تسجيل الدخول لإصلاح خطأ image_4801c2
    public function loginPost(Request $request)
    {
        $credentials = $request->validate(['email' => 'required|email', 'password' => 'required']);
        if (Auth::attempt($credentials)) {
            $request->session()->regenerate();
            return redirect()->route('admin.dashboard');
        }
        return back()->withErrors(['email' => 'بيانات الدخول غير صحيحة.']);
    }

    // دالة إنشاء سؤال لإصلاح خطأ image_481483
    public function create() { return view('admin.questions.create'); }

    // دالتي الملف الشخصي
    public function editProfile() { return view('admin.profile', ['user' => Auth::user()]); }
    public function updateProfile(Request $request) {
        $user = Auth::user();
        $user->name = $request->name;
        $user->email = $request->email;
        if ($request->filled('password')) { $user->password = Hash::make($request->password); }
        $user->save();
        return back()->with('success', 'تم التحديث');
    }

    // باقي دوال الـ Resource (index, store, edit, update, destroy) تظل كما هي في كودك الأصلي
    public function index() {
        $questions = Question::with('answer')->orderBy('sort_order', 'asc')->get();
        return view('admin.questions.index', compact('questions'));
    }

    public function store(Request $request) {
        $question = Question::create(['title' => $request->title, 'sort_order' => $request->sort_order, 'phone' => $request->phone]);
        Answer::create(['question_id' => $question->id, 'content' => $request->content]);
        return redirect()->route('admin.questions.index');
    }
}